#The following R code is for the publication Reddin et al. 2019, Marine clade sensitivities to climate change conform across time scales
#It is structured for navigation in R-Studio
#Code is included (a) for any derived variables not entered directly into the modern dataset, and (b) for all handling stages of the fossil dataset from raw PBDB download
#Alternatively, the main two supporting data files are already prepared for analysis

rm(list=ls())

#Set the working directory to where the files from the repository are saved
setwd("C:/Users/...")

expdat<-read.csv("Reddin et al. Modern metaanalysis supporting data.csv")
slc.dat<-read.csv("Reddin et al. Fossil selectivity supporting data.csv") #Preprepared fossil data

##Packages ----
library(metafor)
library(readxl)
library(seacarb)
library(car)
library(divDyn)
library(glmulti)
library(wCorr)
library(lme4)

#Functions ----
 
slicer<-function(x){stages$stg[x<stages$bottom  &  x>=stages$top]} #Simple age-binning of fossil occurrences (using stages from divDyn package)

#Synthesising across multiple binomial regression models to find the average estimates
Phan.meta.select<-function(outputcols,weight,P){
  taxmeans<-matrix(NA,ncol=6,nrow=length(unique(row.names(outputcols))))
  for (i in 1:(length(unique(row.names(outputcols))) )){
    taxmeans[i,1]<- unique(row.names(outputcols))[i]
    if(weight%in%c("back","ext")){
      if(weight=="back"){
        wt<-1-ext.rate[as.numeric(outputcols[row.names(outputcols)==unique(row.names(outputcols))[i]&as.numeric(outputcols[,3])<=P,4])]}
      if(weight=="ext"){
        wt<-ext.rate[as.numeric(outputcols[row.names(outputcols)==unique(row.names(outputcols))[i]&as.numeric(outputcols[,3])<=P,4])]}
      mod1<-rma(as.numeric(outputcols[row.names(outputcols)==unique(row.names(outputcols))[i]&as.numeric(outputcols[,3])<=P,1]),
                sei=as.numeric(outputcols[row.names(outputcols)==unique(row.names(outputcols))[i]&as.numeric(outputcols[,3])<=P,2]),
                weights=wt,method="REML") }
    if(weight=="none"){
      mod1<-rma(as.numeric(outputcols[row.names(outputcols)==unique(row.names(outputcols))[i]&as.numeric(outputcols[,3])<=P,1]),
                sei=as.numeric(outputcols[row.names(outputcols)==unique(row.names(outputcols))[i]&as.numeric(outputcols[,3])<=P,2]),
                weighted=FALSE,method="REML",control=list(stepadj=0.5,maxiter=100)) } #step adjust needed for Fisher scoring algorithm to converge
    if(weight=="invar"){
      mod1<-rma(as.numeric(outputcols[row.names(outputcols)==unique(row.names(outputcols))[i]&as.numeric(outputcols[,3])<=P,1]),
                sei=as.numeric(outputcols[row.names(outputcols)==unique(row.names(outputcols))[i]&as.numeric(outputcols[,3])<=P,2]),
                method="REML",control=list(stepadj=0.5,maxiter=100)) } 
    taxmeans[i,2]<-mod1$beta
    taxmeans[i,3]<-mod1$ci.lb #fit.stats
    taxmeans[i,4]<-mod1$ci.ub
    taxmeans[i,5]<-mod1$se
    taxmeans[i,6]<-mod1$fit.stats[5,2] } #this is the AICc of the model for the taxon
  taxmeans<-gsub("Taxa","",taxmeans,fixed=T) 
  return(taxmeans)}

#This function selects the responses desired for analysis, by developmental stage, response type, stressor(s), and prioritisation for breaking pseudo-replication -- note the data are already a result of running this function in format, newdat<-BigSubset(NA,NA,NA,"Var") 
BigSubset<-function(Entw.sta,ResponseTypeSimpl,Stresso,Priority,TOpt=0){  
  
  if(is.na(Entw.sta[1])){ rm(Entw.sta)
    Entw.sta<-c("Embryo","Larvae","Juvenile","Adult","Gamete")}
  if(is.na(ResponseTypeSimpl[1])){ rm(ResponseTypeSimpl)
    ResponseTypeSimpl<-c("Calcification","Survival","Fertilization","Development","Growth","Metabolism","Condition")}
  if(is.na(Stresso[1])){ rm(Stresso)
    Stresso<-c("S","T","pH","O","T, S","T, pH","T, O","pH, O","T, pH, S","T, pH, O","T, S, O")}
  
  #Splitting groups (alternative is to use them as moderators)
  dataset.test<-expdat
  #To manually remove 'no variance' tests (but can also subset other items within rma() )
  dataset.test<-subset(dataset.test,is.na(dataset.test$vi)==FALSE & ResponseTypeSimple %in% ResponseTypeSimpl & Entw.stad %in% Entw.sta & Stressor %in% Stresso)
  
  #Study independence columns
  #Code to give priority to, from non-independent experiments, the longest duration, most negative effects size, or smallest variance
  dataset.test$MaxDurat <- FALSE
  dataset.test$MinEffect <- FALSE
  dataset.test$MinVar <- FALSE
  dataset.test$Rand <- FALSE
  
  #Gives binary columns prioritising the non-independent experiments by longest duration (generally the same within
  #a study so probably useless), most negative effect size, or smallest variance. Ties are broken by randomised 
  #sampling (generally extremely rare for effect size or variance)
  for (i in 1:length(unique(dataset.test$id))){
    durat.select<-which(dataset.test[dataset.test$id==unique(dataset.test$id)[i],]$t..d.==max(dataset.test[dataset.test$id==unique(dataset.test$id)[i],]$t..d.))
    eff.select<-which(dataset.test[dataset.test$id==unique(dataset.test$id)[i],]$yi==min(dataset.test[dataset.test$id==unique(dataset.test$id)[i],]$yi))
    var.select<-which(dataset.test[dataset.test$id==unique(dataset.test$id)[i],]$vi==min(dataset.test[dataset.test$id==unique(dataset.test$id)[i],]$vi))
    rand.select<-sample(length(dataset.test[dataset.test$id==unique(dataset.test$id)[i],1]),1)
    if (length(durat.select)>1){ 
      durat.select<-sample(durat.select,1)}
    if (length(eff.select)>1){ 
      eff.select<-sample(eff.select,1)}
    if (length(var.select)>1){ 
      var.select<-sample(var.select,1)}
    dataset.test[dataset.test$id==unique(dataset.test$id)[i],]$MaxDurat[durat.select]<-TRUE
    dataset.test[dataset.test$id==unique(dataset.test$id)[i],]$MinEffect[eff.select]<-TRUE
    dataset.test[dataset.test$id==unique(dataset.test$id)[i],]$MinVar[var.select]<-TRUE
    dataset.test[dataset.test$id==unique(dataset.test$id)[i],]$Rand[rand.select]<-TRUE
  }
  
  if(Priority=="Eff"){
    #Severest effect (contrast results with the most beneficial?)
    dataset.test <- dataset.test[dataset.test$MinEffect,]}
  if(Priority=="Var"){
    #Smallest variance
    dataset.test <- dataset.test[dataset.test$MinVar,]}
  if(Priority=="Rand"){
    #random
    dataset.test <- dataset.test[dataset.test$Rand,]}
  
  if(TOpt==1){
    nichecent<-(dataset.test$TPref.Max.90th+dataset.test$TPref.Min.10th)/2
    temponly<-dataset.test$Stressor %in% c("T","T, O","T, pH","T, S","T, pH, S","T, pH, O","T, S, O")
    print(sum(temponly==T  & is.na(nichecent)==F & dataset.test$T2.m>nichecent,na.rm=T)/sum(temponly==T & is.na(nichecent)==F));print(sum(is.na(nichecent)==F)/length(nichecent))
    dataset.test<-dataset.test[temponly==T & is.na(nichecent)==F & dataset.test$T2.m>nichecent,] }
  return(dataset.test)}

##Calculating PP using range completeness
samplcompl<-function(tax){   
  tempFL <- tapply(INDEX = spdat[, tax], X = 1:nrow(spdat), 
                   FUN = function(x) {
                     suppressWarnings(c(min(spdat[x, "slc"], na.rm = T), 
                                        max(spdat[x, "slc"], na.rm = T), length(unique(spdat[x, "slc"])))) })
  fl <- matrix(unlist(tempFL), ncol = 3, byrow = T)
  fl[!is.finite(fl)] <- NA
  fl <- as.data.frame(fl)
  colnames(fl) <- c("FAD", "LAD","Sampl")
  fl$LAD<-95 #this is added to show that these organisms survive until today
  fl$range <- fl[, "LAD"] - fl[, "FAD"]
  fl <- fl[, c("FAD", "LAD","Sampl", "range")]
  fl$SamCom <- fl[, "Sampl"] / (fl[, "range"]+1)
  fl <- fl[, c("FAD", "LAD","Sampl", "range","SamCom")]
  nullnam<-lapply(tempFL,is.null) #added
  rownames(fl) <- names(tempFL)[nullnam==F] 
  fl$rangSam<-fl$range*fl$SamCom
  fl<-fl[-1,]
  if(tax=="genus"){
    fl<-fl[rownames(fl) %in% expdat$Genus,]}
  if(tax=="family"){
    fl<-fl[rownames(fl) %in% expdat$Family,]}
  if(tax=="order"){
    fl<-fl[rownames(fl) %in% expdat$Order,]}
  if(tax=="class"){
    fl<-fl[rownames(fl) %in% expdat$ClassFossil,]}
  return(fl)
}
 
#Correlation of differences for time series
gen.diff.corr<-function(x,z){
  f1<-cor.test(x[-1],x[-length(x)]) #First order correlation
  new.f1<-x[-1]-f1$estimate*x[-length(x)] #Generalised differences temp
  f2<-cor.test(z[-1],z[-length(z)]) #First order correlation
  new.f2<-z[-1]-f2$estimate*z[-length(z)] #Generalised differences temp
  out<-cor.test(new.f1,new.f2,method="spearman")
  return(out)
}
 
#Rho confidence intervals
 rho.CIs<-function(x,n){
  rho.CI.UL<-c(tanh(atanh(x)-(1.96/sqrt(n-3))),tanh(atanh(x)+(1.96/sqrt(n-3))))
  return(rho.CI.UL)
 }
 
 #Plotting function to compare selectivity at individual geological ages with modern experimental sensitivity
 eventplot<-function(i,eventit,oddsfile){
   tit<-paste("Fossil model pseudo-R","=",round(as.numeric(modloglik[i,1]),digits = 2)); tit2<-paste(eventit) 
   intrank<-oddsfile
   lims<-c(min((as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1)-(1.96*as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),2])))),max((as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1)+(1.96*as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),2])))))
   
   CRSbothweight<-(scale((1/as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),2]))),center=F) +
                     scale(as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),4])),center=F))/2
   plot(as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1,
        as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),1])),ylab="Experimental CRS sensitivity, Hedge's D",
        xlab="Survival selectivity, log-odds",main=paste(tit2),pch=1,xlim=lims,ylim=c(-1.5,1),cex=CRSbothweight*2)        
   if(sum(rownames(intrank) == "Gastropoda")>0){
     text(as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals)[c(1:6,8:10)],1]))*-1,
          as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank)[!rownames(intrank)=="Gastropoda"],1])),labels=rownames(intrank[rownames(intrank) %in% rownames(CRSExptVals)[c(1:6,8:10)],]),adj=c(-0.1,1))
     text(as.numeric(as.vector(intrank[rownames(intrank) == "Gastropoda",1]))*-1,
          as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) == "Gastropoda",1])),labels="Gastropoda",adj=c(0.5,-1))}
   else{  text(as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1,
               as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),1])),labels=rownames(intrank[rownames(intrank) %in% rownames(CRSExptVals),]),adj=c(-0.1,1))}
   arrows((as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1)+(1.96*as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),2]))),
          as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),1])),
          (as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1)-(1.96*as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),2]))),
          as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),1])),code = 3,angle=90,length=0,col="darkgrey") 
   arrows(as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1,
          as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),2])),
          as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1,
          as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),3])),code = 3,angle=90,length=0,col="darkgrey")
   
   CRScorr<-round(weightedCorr(
     rank(as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1),
     rank(as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),1]))) ,method="spearman",
     weights =CRSbothweight),digits = 2)
   
   TpHbothweight<-(scale(1/as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(TpHExptVals),2])),center=F) +
                     scale(as.numeric(as.vector(TpHExptVals[rownames(TpHExptVals) %in% rownames(intrank),4])),center=F))/2
   TpHcorr<-round(weightedCorr(
     rank(as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(TpHExptVals),1]))*-1),
     rank(as.numeric(as.vector(TpHExptVals[rownames(TpHExptVals) %in% rownames(intrank),1]))) ,method="spearman",
     weights =TpHbothweight),digits = 2)
   
   Obothweight<-(scale(1/as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(OExptVals),2])),center=F) +
                   scale(as.numeric(as.vector(OExptVals[rownames(OExptVals) %in% rownames(intrank),4])),center=F))/2
   Ocorr<-round(weightedCorr(
     rank(as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(OExptVals),1]))*-1),
     rank(as.numeric(as.vector(OExptVals[rownames(OExptVals) %in% rownames(intrank),1]))) ,method="spearman",
     weights =Obothweight),digits = 2)
   legend("topleft",legend=c(paste("Overall, Rho = ",CRScorr," (",paste0(round(rho.CIs(CRScorr,length(intrank[rownames(intrank) %in% rownames(CRSExptVals),1])),digits = 2),collapse="-"),")",sep=""),
                             paste("T & pH, Rho = ",TpHcorr," (",paste0(round(rho.CIs(TpHcorr,length(intrank[rownames(intrank) %in% rownames(TpHExptVals),1])),digits = 2),collapse="-"),")",sep=""),
                             paste("O, Rho = ",Ocorr)),bty="n")
   legend("bottomright",legend=tit,bty="n")}


#Preparing the fossil data from original PBDB download ----
dat <- read.csv(file="pbdb_data.csv", header=TRUE,skip=19) 

data(stages) #from divDyn

dat$mean_ma<-(dat$max_ma+dat$min_ma)/2
spdat<-dat[!(dat$max_ma-dat$min_ma)>19.5,] #19.5MY is the longest geological age (Norian)

#Remove early, middle and late, and remaining any spaces
x<-spdat$early_interval
x<-gsub("Early", "", x) 
x<-gsub("Middle", "", x) 
x<-gsub("Late", "", x) 
x<-gsub(" ", "", x)
spdat$early_interval<-x

#Combining class and genus is more robust to homonyms
spdat$clgen<- paste(spdat$class, spdat$genus)

# Delete duplicate genus occurrences within collections
t1 <- subset(spdat, select=c(collection_no, clgen))
spdat <- spdat[!duplicated(t1),]

#Slicer for ages (lowest priority)
spdat$slc<-lapply(spdat$mean_ma,slicer)
spdat$slc<-as.numeric(spdat$slc)

#Use early_interval where possible and where the only interval supplied (highest priority)
for (i in 14:95){
  spdat$slc[spdat$early_interval %in% stages$stage[i] & spdat$late_interval==""| spdat$late_interval==spdat$early_interval]<-i}

spdat <- subset(spdat, is.na(slc)==F) # Exclude unassigned bins (should be zero anyway)

#Add Forams to class so they don't get lost in next steps
levels(spdat$class) <- c(levels(spdat$class),"Foraminifera")
spdat[spdat$phylum=="Foraminifera",]$class<-paste("Foraminifera") #upgrading forminifera to 'class' level in fossil record (so that it is caught by recent taxa)

#check marine and relatively well-sampled or common groups
table(spdat$class)[order(table(spdat$class))] #excluding mammals, insects, reptiles, classes <1000 occurrences
cl.n <- c("Mammalia","", "Insecta","Collembola","Reptilia","Pteridopsida","Pinopsida","Aves","Craniata","Lycopsida","Saurischia","Ornithischia","Cycadopsida","Arachnida","Ginkgoopsida","Equisetopsida","Cycadeoideopsida","Myriapoda","Peltaspermopsida","Archaeopteridopsida","Psilophytopsida","Dictyopteridiopsida","Zosterophyllopsida","Aneurophytopsida","Pteridospermopsida","Scorpionida")
spdat <- subset(spdat, !class %in% cl.n)

#Large numbers without a class, so better to look at their phyla and drop blanks for known terrestrial groups
ph.n <- c("","Lycopodiophyta","Problematica","Bryophyta","Tracheophyta","Pteridophyta","Angiospermae","Cycadophyta","Spermatophyta","Lycophyta","Pinophyta","Magnoliophyta","Onychophora")
spdat <- subset(spdat, !phylum %in% ph.n | class%in% c("Rhodophyceae","Foraminifera"))
spdat <- subset(spdat, !order %in% c("Temnospondyli","Cotylosauria")) #removing a couple more amphibian or reptile groups

spdat$class[spdat$class%in%c("Osteichthyes","Actinopteri")]<-paste("Actinopterygii") 
colnames(spdat)[colnames(spdat)%in%c("paleolng","paleolat")]<-c("paleolngdec","paleolatdec")
spdat <- subset(spdat, is.na(paleolatdec)==F & paleolatdec!=0) #removed unrotated occurrences
spdat<-spdat[!spdat$genus%in%c(""," "),] #remove unlabelled genera
spdat[spdat$order=="","order"]<-NA #and orders
#correcting a couple of bad data (now fixed in the PBDB)
spdat<-spdat[!(spdat$clgen%in% c("Anthozoa Paleofavosites","Anthozoa Ainia") &spdat$slc>51),] #Triassic tabulate and fish homonym

#Removing singletons 
x <- table(factor(spdat$clgen))
spdat <- subset(spdat, !spdat$clgen %in% names(x[x==1]))

#Forming the raw table for binomial regression, of extinct/survival per genus per age and categorical predictors
slc.dat<-modeltab(spdat, "clgen", "slc", taxvars = c("phylum","class","genus","order","composition","motility","diet","life_habit","paleolatdec","paleolngdec"), rt = TRUE, singletons = TRUE)
names(slc.dat)[5:9]<-c("Slc","Phylum","Class","Genus","Order")

#Populating 'Taxa' (the predictor category 'clade') wih higher taxa
slc.dat$Taxa<-as.character(slc.dat$Phylum)
slc.dat$Taxa[slc.dat$Phylum%in%"Mollusca"]<-paste(slc.dat$Class[slc.dat$Phylum%in%"Mollusca"])
slc.dat$Taxa[slc.dat$Phylum%in%"Chordata"]<-paste(slc.dat$Class[slc.dat$Phylum%in%"Chordata"])
slc.dat$Taxa[slc.dat$Phylum%in%"Echinodermata"]<-paste(slc.dat$Class[slc.dat$Phylum%in%"Echinodermata"])
slc.dat$Taxa[slc.dat$Phylum%in%"Cnidaria"]<-paste(slc.dat$Class[slc.dat$Phylum%in%"Cnidaria"])
slc.dat$Taxa[slc.dat$Phylum%in%"Arthropoda"]<-paste(slc.dat$Class[slc.dat$Phylum%in%"Arthropoda"])
slc.dat$Taxa[slc.dat$Phylum%in%"Porifera"]<-paste(slc.dat$Class[slc.dat$Phylum%in%"Porifera"])
slc.dat$Taxa[slc.dat$Taxa%in%c("Cirripedia","Branchiopoda","Malacostraca","Maxillopoda")]<-paste("Crustacean") 
slc.dat$Taxa[slc.dat$Class%in%"Bivalvia"]<-paste(slc.dat$Order[slc.dat$Class%in%"Bivalvia"])

##Handling synonymous fossil functional groups ----
#Motility
slc.dat$motility[slc.dat$Phylum=="Porifera"]<-paste("stationary, attached")
slc.dat$motility[slc.dat$motility%in%c("stationary, attached, epibiont","stationary, epibiont","stationary, attached","attached, epibiont")]<-paste("stationary")
slc.dat$motility[slc.dat$motility%in%c("facultatively mobile, attached","facultatively mobile, epibiont")]<-paste("facultatively mobile")
slc.dat$motility[slc.dat$motility=="passively mobile, epibiont"]<-paste("passively mobile")
slc.dat$motility[slc.dat$motility%in%c("fast-moving","slow-moving")]<-paste("actively mobile")
slc.dat$motility[slc.dat$motility==""]<-NA
#Primary skeletal composition
slc.dat$composition[slc.dat$composition%in%c("intermediate Mg calcite","intermediate Mg calcite, aragonite","high Mg calcite, aragonite")]<-paste("high Mg calcite")
slc.dat$composition[slc.dat$composition%in%c("low Mg calcite, aragonite","low Mg calcite, chitin")]<-paste("low Mg calcite")
slc.dat$composition[slc.dat$composition%in%c("aragonite, low Mg calcite","aragonite, \"calcite\"","aragonite, intermediate Mg calcite")]<-paste("aragonite")
slc.dat$composition[slc.dat$composition%in%c("","\"calcite\", aragonite","\"calcite\"")]<-NA
#Life habit
slc.dat$life_habit[slc.dat$life_habit%in%c("deep infaunal","semi-infaunal, gregarious","shallow infaunal","semi-infaunal, solitary","semi-infaunal","infaunal, depth=deep")]<-paste("infaunal")
slc.dat$life_habit[slc.dat$life_habit%in%c("nektobenthic","nektonic, depth=surface","nektonic, depth=deep","nektobenthic, solitary","nektonic, solitary")]<-paste("nektonic")
slc.dat$life_habit[slc.dat$life_habit%in%c("planktonic, depth=surface","planktonic, solitary")]<-paste("planktonic")
slc.dat$life_habit[slc.dat$life_habit%in%c("amphibious","","aquatic","colonial","solitary, clonal","solitary")]<-NA
slc.dat$life_habit[slc.dat$life_habit%in%c("intermediate-level epifaunal","low-level epifaunal","upper-level epifaunal","intermediate-level epifaunal, colonial, clonal","low-level epifaunal, solitary","epifaunal, solitary","epifaunal, gregarious","epifaunal, clonal","colonial, clonal","colonial, clonal, depth=subthermocline","low-level epifaunal, solitary, clonal","intermediate-level epifaunal, gregarious","intermediate-level epifaunal, solitary, clonal","epifaunal, solitary, depth=subthermocline","epifaunal, polymorph","epifaunal, colonial, clonal","epifaunal, gregarious, clonal")]<-paste("epifaunal")
slc.dat$life_habit[slc.dat$life_habit==""]<-NA
#Diet
slc.dat$diet[slc.dat$diet%in%c("microcarnivore, photosymbiotic","suspension feeder, deposit feeder","suspension feeder, detritivore","suspension feeder, photosymbiotic","suspension feeder, chemosymbiotic","suspension feeder, carnivore")]<-paste("suspension feeder")
slc.dat$diet[slc.dat$diet%in%c("grazer, deposit feeder","detritivore","deposit feeder, suspension feeder","deposit feeder, suspension feeder","detritivore, suspension feeder","coprophage, deposit feeder","deposit feeder, chemosymbiotic","deposit feeder, detritivore","detritivore, grazer")]<-paste("deposit feeder")
slc.dat$diet[slc.dat$diet%in%c("omnivore, grazer","grazer, omnivore")]<-paste("omnivore")
slc.dat$diet[slc.dat$diet%in%c("photosymbiotic, microcarnivore","photosymbiotic","photosymbiotic, suspension feeder")]<-paste("photosymbiotic")
slc.dat$diet[slc.dat$diet%in%c("microcarnivore","carnivore, deposit feeder","carnivore, detritivore","carnivore, parasite","carnivore, piscivore","carnivore, suspension feeder","durophage, carnivore","microcarnivore, carnivore")]<-paste("carnivore")
slc.dat$diet[slc.dat$diet%in%c("herbivore","grazer, detritivore")]<-paste("grazer")
slc.dat$diet[slc.dat$diet%in%c("","chemosymbiotic","chemosymbiotic, suspension feeder","coprophage","durophage","parasite, carnivore","parasite")]<-NA
slc.dat$diet[slc.dat$diet==""]<-NA
#Buffering TRUE/FALSE (following Table 2 from Bambach et al. 2002)
slc.dat$buffered<-NA #Forams left out
slc.dat$buffered[slc.dat$Order%in%c("Actiniaria","Nautiloidea","Eurypterida","Conodontophorida","Textulariina")|slc.dat$class%in%c("Actinopterygii","Malacostraca","Chondrichthyes","Polychaeta","Gastropoda","Acanthodii","Agnatha","Asteroidea","Ophiuroidea")]<-TRUE
slc.dat$buffered[slc.dat$Order%in%c("Scleractinia","Cheilostomatida","Ammonoidea","Belemnitida","Favositida","Heliolitida","Auloporida")|slc.dat$Class%in%c("Echinoidea","Rhynchonellata","Strophomenata","Chileata","Stenolaemata","Ostracoda","Rhombifera","Blastoidea","Crinoidea","Graptolithina")]<-FALSE
slc.dat$buffered[slc.dat$Class=="Bivalvia" & slc.dat$life_habit=="infaunal"]<-TRUE
slc.dat$buffered[slc.dat$Class=="Bivalvia" & slc.dat$life_habit=="epifaunal"]<-FALSE
slc.dat$buffered[slc.dat$Class=="Demospongiae" & slc.dat$composition=="calcareous"]<-FALSE 
slc.dat$buffered[slc.dat$Class=="Demospongiae" & slc.dat$composition%in%c("organic","siliceous")]<-TRUE 

##Regression per interval ----
#Matrices for the separate regressions per grouping type and geological age
slc.dat<-slc.dat[!slc.dat$Slc==95,]
tax.sig.all<-matrix(NA,nrow=2,ncol=4)
comp.sig.all<-matrix(NA,nrow=2,ncol=4)
mot.sig.all<-matrix(NA,nrow=2,ncol=4)
diet.sig.all<-matrix(NA,nrow=2,ncol=4)
lifhab.sig.all<-matrix(NA,nrow=2,ncol=4)
buff.sig.all<-matrix(NA,nrow=2,ncol=4)
#And vectors for the model log likelihoods
modloglik<-numeric(length=95)
modloglik2<-numeric(length=95)
modloglik3<-numeric(length=95)
modloglik4<-numeric(length=95)
modloglik5<-numeric(length=95)
modloglik6<-numeric(length=95)

b<-Sys.time()
nulldf<-numeric(length=95)
for (i in unique(slc.dat$Slc)[order(unique(slc.dat$Slc))]){ #Be patient. This takes quite a while... (an hour on quite a good computer)
  subsdat<-subset(slc.dat,slc.dat$Slc==i)
  subsdat<-subsdat[subsdat$Taxa %in% names(table(subsdat$Taxa)[table(subsdat$Taxa)>6]),] #Min 6 genera per class, since this is the minimum needed to be significant e.g. binom.test(6,6) vs binom.test(5,5)
  m1<-glm(ext~Taxa-1,family="binomial",data=subsdat[is.na(subsdat$ext)==F,]) 
  nullmod <- glm(ext~1, family="binomial",data=subsdat[is.na(subsdat$ext)==F,])
  modloglik[i]<-1-logLik(m1)/logLik(nullmod)
  nulldf[i]<-m1$df.null
  
  m2<-glmer(ext~composition + (composition | Taxa) -1,family="binomial",data=subsdat[!is.na(subsdat$ext) & subsdat$composition%in%c("aragonite", "high Mg calcite",  "hydroxyapatite",  "low Mg calcite"),]) 
  m3<-glmer(ext~motility+ (motility | Taxa)-1,family="binomial",data=subsdat[!is.na(subsdat$ext)& subsdat$motility%in%c("actively mobile","stationary"),])
  m4<-glmer(ext~diet+ (diet | Taxa)-1,family="binomial",data=subsdat[!is.na(subsdat$ext)& subsdat$diet%in%c("carnivore","deposit feeder","grazer","omnivore","suspension feeder"),]) 
  m5<-glmer(ext~life_habit+ (life_habit | Taxa)-1,family="binomial",data=subsdat[!is.na(subsdat$ext)& subsdat$life_habit%in%c("epifaunal","infaunal","nektonic"),]) 
  if(isSingular(m2)){print(c(stages$stage[i],"+composition"))} #printing which models have singular fits of random effects
  if(isSingular(m3)){print(c(stages$stage[i],"+motility"))}
  if(isSingular(m4)){print(c(stages$stage[i],"+diet"))}
  if(isSingular(m5)){print(c(stages$stage[i],"+life_habit"))}
  
  modloglik2[i]<-1-logLik(m2)/logLik(nullmod) #McFadden's pseudo-R squared
  modloglik3[i]<-1-logLik(m3)/logLik(nullmod)
  modloglik4[i]<-1-logLik(m4)/logLik(nullmod)
  modloglik5[i]<-1-logLik(m5)/logLik(nullmod)
  
  if(sum(summary(m1)$coefficients[2:length(summary(m1)$coefficients[,1]),4]<0.1)>0){
    tax.sig<-cbind(summary(m1)$coefficients[,c(1,2,4)][row.names(summary(m1)$coefficients)%in%c("TaxaEchinoidea","TaxaActinopterygii","TaxaAnthozoa","TaxaAsteroidea","TaxaBivalvia","TaxaBrachiopoda","TaxaCrustacean","TaxaGastropoda","TaxaOphiuroidea","TaxaPorifera")|summary(m1)$coefficients[,4]<0.2 ,],i) 
    tax.sig.all<-rbind(tax.sig.all,tax.sig)} 
  if(sum(summary(m2)$coefficients[2:length(summary(m2)$coefficients[,1]),4]<0.99)>0){
    tax.sig<-cbind(summary(m2)$coefficients[,c(1,2,4)][row.names(summary(m2)$coefficients)%in%c("compositionaragonite", "compositionhigh Mg calcite",  "compositionhydroxyapatite",  "compositionlow Mg calcite") ,],i) 
    comp.sig.all<-rbind(comp.sig.all,tax.sig)} 
  if(sum(summary(m3)$coefficients[2:length(summary(m3)$coefficients[,1]),4]<0.99)>0){
    tax.sig<-cbind(summary(m3)$coefficients[,c(1,2,4)][row.names(summary(m3)$coefficients)%in%c("motilityactively mobile","motilitystationary") ,],i) 
    mot.sig.all<-rbind(mot.sig.all,tax.sig)} 
  if(sum(summary(m4)$coefficients[2:length(summary(m4)$coefficients[,1]),4]<0.99)>0){
    tax.sig<-cbind(summary(m4)$coefficients[,c(1,2,4)][row.names(summary(m4)$coefficients)%in%c("dietcarnivore","dietdeposit feeder","dietgrazer","dietomnivore","dietsuspension feeder") ,],i) 
    diet.sig.all<-rbind(diet.sig.all,tax.sig)} 
  if(sum(summary(m5)$coefficients[2:length(summary(m5)$coefficients[,1]),4]<0.99)>0){
    tax.sig<-cbind(summary(m5)$coefficients[,c(1,2,4)][row.names(summary(m5)$coefficients)%in%c("life_habitepifaunal","life_habitinfaunal","life_habitnektonic") ,],i) 
    lifhab.sig.all<-rbind(lifhab.sig.all,tax.sig)}
  
  if(length(unique(subsdat$buffered[is.na(subsdat$buffered)==F]))>1){
    m6<-glmer(ext~factor(buffered)+ (factor(buffered) | Taxa)-1,family="binomial",data=subsdat[!is.na(subsdat$ext)& !subsdat$buffered=="",]) 
    modloglik6[i]<-1-logLik(m6)/logLik(nullmod)
    
    if(sum(summary(m6)$coefficients[2:length(summary(m6)$coefficients[,1]),4]<0.99)>0){
      tax.sig<-cbind(summary(m6)$coefficients[,c(1,2,4)],i) 
      buff.sig.all<-rbind(buff.sig.all,tax.sig)}}
} #ignoring singular fits of random effects as we focus on the fixed effects
Sys.time()-b

#Remove empty rows
tax.sig.all<-tax.sig.all[-1:-2,]
comp.sig.all<-comp.sig.all[-1:-2,]
mot.sig.all<-mot.sig.all[-1:-2,]
diet.sig.all<-diet.sig.all[-1:-2,]
lifhab.sig.all<-lifhab.sig.all[-1:-2,]
buff.sig.all<-buff.sig.all[-1:-2,]

#To add stage names to the model outputs (for easier checking)
st.names.tax<-numeric();st.names.comp<-numeric();st.names.mot<-numeric();st.names.diet<-numeric();st.names.lifhab<-numeric();st.names.buff<-numeric()
for (i in 14:94){
  st.names.tax[tax.sig.all[,4]==i]<-paste(stages[i,"stage"])
  st.names.comp[comp.sig.all[,4]==i]<-paste(stages[i,"stage"])
  st.names.mot[mot.sig.all[,4]==i]<-paste(stages[i,"stage"])
  st.names.diet[diet.sig.all[,4]==i]<-paste(stages[i,"stage"])
  st.names.lifhab[lifhab.sig.all[,4]==i]<-paste(stages[i,"stage"])
  st.names.buff[buff.sig.all[,4]==i]<-paste(stages[i,"stage"])}
tax.sig.all<-cbind(tax.sig.all,st.names.tax)
comp.sig.all<-cbind(comp.sig.all,st.names.comp)
mot.sig.all<-cbind(mot.sig.all,st.names.mot)
diet.sig.all<-cbind(diet.sig.all,st.names.diet)
lifhab.sig.all<-cbind(lifhab.sig.all,st.names.lifhab)
buff.sig.all<-cbind(buff.sig.all,st.names.buff)
#To add stage names to the model McFadden's pseudo-R squared
modloglik<-cbind(modloglik,stages[,4])
modloglik2<-cbind(modloglik2,stages[,4])
modloglik3<-cbind(modloglik3,stages[,4])
modloglik4<-cbind(modloglik4,stages[,4])
modloglik5<-cbind(modloglik5,stages[,4])
modloglik6<-cbind(modloglik6,stages[,4])
nulldf<-cbind(nulldf,stages[,4]) 

#Synthesise the models over the Phanerozoic with inverse-variance weighting of ages
taxmedians<-Phan.meta.select(tax.sig.all,"invar",0.95) #Note: P filtering is theoretically not necessary because of inverse-variance weighting, but models do not converge without some restriction
overallcompext<-Phan.meta.select(comp.sig.all[!as.numeric(comp.sig.all[,3])==0,],"invar",0.95)
overallmotext<-Phan.meta.select(mot.sig.all[!mot.sig.all[,3]==0,],"invar",0.95) 
overalldietext<-Phan.meta.select(diet.sig.all[!diet.sig.all[,3]== 0,],"invar",0.95)
overalllifhabext<-Phan.meta.select(lifhab.sig.all[!row.names(lifhab.sig.all)=="life_habitinfaunal" & !lifhab.sig.all[,3]== 0,],"invar",0.95) #infauna removed since these are not well-represented by modern estimates 
overallbuffext<-Phan.meta.select(buff.sig.all,"invar",0.95)

#For display of the results, traits in rank order
overallcompext[order(as.numeric(overallcompext[,2])),]
overallmotext[order(as.numeric(overallmotext[,2])),]
overalldietext[order(as.numeric(overalldietext[,2])),]
overalllifhabext[order(as.numeric(overalllifhabext[,2])),]
overallbuffext


##Preparing the modern data from the raw dataset ----

##Categorising responses types ----
#Note: the order is important since the later (e.g. Survival) take priority over the earlier (e.g. Metabolism)
#Metabolism or aerobic scope 
expdat$ResponseTypeSimple[expdat$`Response type`%in% c("respiration","respiration rates" ,"respiration rate", "swimming activity","O2 consumption","Ammonia excretion","Respiration","Absorptionsrate","oxygen consumption","Oxygen consumption","Resting MO2","Tenacity","Metabolism" , "Mean O2 uptake","feeding rate","consumption of U. rigida","inactive metabolic rate","maximum metabolic rate","active metabolic rate","routine metabolic rate","Maximum MO2", "Mean critical O2", "thermal tolerance (measured through PaO2 values in the haemolymph)","Aerobic scope","oxygen uptake","resting metabolic rate","oxygen extraction","gill ventilation","heart rate" , "movement" , "oxidative metabolism & immune system" , "carbohydrate metabolism" , "lipid metabolism", "respiratory frequency", "cardiac frequency","cardiac stroke volume" ,"cardiac output" ,"mean pressure in the ventral aorta" ,"mean pressure in the dorsal aorta", "lethal oxygen level", "maximal oxygen consumption", "maximal heart rate", "standard metabolic rate","mean oxygen consumption","NH3 excretion","Urate","critical O2 before emersion","reversal frequency","reversal duration","critical oxygen concentration","swimming metabolic rate","aerobic metabolic rate","slope of power-performance curve","succinate concentration gills","succinate concentration hepatopancreas","Whole body lactate","clearance rate","aerobic scope","Ctmax","maximum oxygen uptake","daily feeding rate","larval swimming activity","burrow length","lethal oxygen concentration (50%)","MO2rest","succinate","ETS activity","alanine","dark respiration") ]<-"Metabolism"
#calcification
expdat$ResponseTypeSimple[expdat$`Response type`%in% c("calcification","shell surface","TLC (tot. length of calcite rods)","Shell length","Calcification (bw)","Spine length","Hardness","Elasticity" ,"Calcification" , "calcified mass","shell length","calcium content","skeletal mass","Shell weight","Average lateral shell length", "Average spiral height" , "Lateral shell length (micron)" , "Spiral height (micron)","shell calcium","calcification rate","completed shell repair","further shell growth after repair","individuals that completed shell repair","pallet length")]<-"Calcification"
#growth
expdat$ResponseTypeSimple[expdat$`Response type`%in% c("Size (Scope of growth)","growth","Growth rate","Stomach volume (proxy for growth","shell surface","Arm length","Shell length","Growth (ww)","Mass","Tissue weight","Growth" , "Size","relative growth","shell length","size","biomass","skeletal mass","dry weight","specific growth rate","growth rate","weight","carapace width","weight specific growth rate","Average lateral shell length", "Average spiral height" , "Lateral shell length (micron)" , "Spiral height (micron)","mean length", "growth rate in weight","adult body size (male)","adult body size (female)","growth rate in length","scope for growth","Early larval growth","Late larval growth","Post-settlement growth rate","growth rate (individuals < 3mm)","growth rate (individuals > 3mm)","pallet length","morphlogy (postoral : midline body length)","growth (weight gained during experiment)")  ]<-"Growth"
#fertilization, includes "sperm motility" and avoiding habitats occupied by parents (authors link to inbreeding)
expdat$ResponseTypeSimple[expdat$`Response type`%in% c("Fertilisation" ,"fertilization" , "fertilisation","sperm motility","sperm swimming speed", "fertilised females","unfertilised females" ,"fecundity" ,"parental preference vs. seawater", "conspecific non-parental preference vs. seawater", "conspecific non-parent vs. parent")  ]<-"Fertilization" 
#Condition (includes reproductive investment, predation success and stress repsonses). 
expdat$ResponseTypeSimple[expdat$`Response type`%in% c("cell specific density","Absorption efficiency","Necrosis","Condition index","Condition" , "Whole body lactate","carbon content","colony condition","cell quota N","cell quota C","cell quota P","C:N","C:P","N:P","bleaching","parasite infection","weighted intensity of parasite infection", "reproductive investment" , "haemocyte mortality" , "phagocytosis" , "Esterase" , "Reactive oxygen species" , "lipofuscin content" , "labilisation period" , "nucleic acid regulation" , "protein regulation" , "stress response" , "HSP70 digestive gland (stress response)" , "HSP70 gills (stress response)" , "metallothionine digestive gland", "metallothionine gills","Lipid index","O:N ratio","Lactate","lipid content","metal sequestration","total anoxidant capacity","Lipid peroxidation","destabilised lyosomes","Predation success (%)","unsaturated neutral lipid content of the digestive cells","lipofuscin content of digestive cells","attachment by day 24","tissue necrosis","tissue bleached")]<-"Condition"
#Development 
expdat$ResponseTypeSimple[expdat$`Response type`%in% c("gastrulation","embryo cleaving","Hatching","settlement successful","settlement unsuccessful","Hatching rate","Abnormal Embryos","Settlement","Development","Duration of hatching","hatchability","First hatching","Normal development","settlement","Hatching time", "D-veligers","hatching","normal larvae","larval length","D-Veliger stage","Abnormality","abnormality","Instar 3 intermoult duration","Metamorphosis","Velum" , "Statocyst", "Diverticula", "Operculum", "Eyespots", "Protoconch", "Arm asymmetry","percentage normal","normal cleavage","normal blastulae", "developmental stage", "undeveloped eggs", "dead broods", "females with dead brood", "dead eggs","eggs hatched","number of eggs","eggs per mm3 body volume","metamorphosis","Larval deformities","eggs hatched by day 2","Early larval growth","Late larval growth","Post-settlement growth rate","larval survival","boring larvae","morphlogy (normal prism)","morphlogy (normal echinoplutei)","morphlogy (postoral arm difference)","morphlogy (postoral : midline body length)")  ]<-"Development"
#Survival (includes homing success on suitable habitats)
expdat$ResponseTypeSimple[expdat$`Response type`%in% c("Survival" ,"mortality" ,"survival","median survival time [h]","post settlement mortality","days until 50% mortality","Mortality","larval survival","adult survival","abundance","population growth rate","response to olfactory cues from Melaleuca","response to olfactory cues from grass" , "response to olfactory cues from Anemones") ]<-"Survival"
  
#ClassFossil is used for consistent naming between modern and fossil clades
expdat$ClassFossil<-expdat$Class
levels(expdat$ClassFossil)<-c(levels(expdat$ClassFossil),paste(c("Foraminifera","Chondrichthyes","Cirripedia")))
expdat$ClassFossil[expdat$Class =="Elasmobranchii"]<-paste("Chondrichthyes")
expdat$ClassFossil[expdat$Class =="Hexanauplia"]<-paste("Cirripedia")
expdat$ClassFossil[expdat$Class %in%c("Tubothalamea","Globothalamea")]<-paste("Foraminifera")

  #Calculating PP using range completeness ----
#First get the completeness for each taxon (in a separate matrix) and then spread them into the tables. Genus and family have been hashed out because order and class are the only functional examples here
gensamcom<-samplcompl("genus")
famsamcom<-samplcompl("family")
ordsamcom<-samplcompl("order")
clasamcom<-samplcompl("class")
expdat$Gensamcom<-NA
for (i in 1:length(unique(expdat$Genus))){
  if (unique(expdat$Genus)[i] %in% rownames(gensamcom)){
    expdat$Gensamcom[expdat$Genus == unique(expdat$Genus)[i]]<-gensamcom$SamCom[rownames(gensamcom)==unique(expdat$Genus)[i]]
  }else(expdat$Gensamcom[expdat$Genus == unique(expdat$Genus)[i]]<-0)}
expdat$Famsamcom<-NA
for (i in 1:length(unique(expdat$Family))){
  if (unique(expdat$Family)[i] %in% rownames(famsamcom)){
    expdat$Famsamcom[expdat$Family == unique(expdat$Family)[i]]<-famsamcom$SamCom[rownames(famsamcom)==unique(expdat$Family)[i]]
  }else(expdat$Famsamcom[expdat$Family == unique(expdat$Family)[i]]<-0)}
expdat$Ordsamcom<-NA
for (i in 1:length(unique(expdat$Order))){
  if (unique(expdat$Order)[i] %in% rownames(ordsamcom)){
    expdat$Ordsamcom[expdat$Order == unique(expdat$Order)[i]]<-ordsamcom$SamCom[rownames(ordsamcom)==unique(expdat$Order)[i]]
  }else(expdat$Ordsamcom[expdat$Order == unique(expdat$Order)[i]]<-0)}
expdat$Clasamcom<-NA
for (i in 1:length(unique(expdat$ClassFossil))){
  if (unique(expdat$ClassFossil)[i] %in% rownames(clasamcom)){
    expdat$Clasamcom[expdat$ClassFossil == unique(expdat$ClassFossil)[i]]<-clasamcom$SamCom[rownames(clasamcom)==unique(expdat$ClassFossil)[i]]
  }else(expdat$Clasamcom[expdat$ClassFossil == unique(expdat$ClassFossil)[i]]<-0)}

  #Adjusting the thermal 'optimum' for assumed seasonality (example code -- TPref.Min.10th is now removed) at higher latitudes
  nichecent<-(expdat$TPref.Max.90th+expdat$TPref.Min.10th)/2
expdat$Tcontprox[is.na(nichecent)==F]<-expdat$T1.m[is.na(nichecent)==F]-(lm(expdat$TPref.Max.90th[is.na(nichecent)==F]~expdat$T1.m[is.na(nichecent)==F])$residuals+mean(expdat$TPref.Max.90th[is.na(nichecent)==F]))
expdat$Ttreatprox[is.na(nichecent)==F&is.na(expdat$T2.m)==F]<-expdat$T2.m[is.na(nichecent)==F&is.na(expdat$T2.m)==F]-(lm(expdat$TPref.Max.90th[is.na(nichecent)==F&is.na(expdat$T2.m)==F]~expdat$T2.m[is.na(nichecent)==F&is.na(expdat$T2.m)==F])$residuals+mean(expdat$TPref.Max.90th[is.na(nichecent)==F&is.na(expdat$T2.m)==F]))


# Beginning of the modern analysis ----

#All independent CRS treatments ready for statistical analysis
newdat<-BigSubset(NA,NA,NA,"Var") 
Stressor1<-newdat$Stressor
levels(Stressor1)<-c(levels(Stressor1),"Trio")
Stressor1[Stressor1 %in% c("T, pH, O","T, pH, S","T, S, O")]<-"Trio" #CRS categories with all three-way CRS responses combined

levels(newdat$ClassFossil)<-c(levels(newdat$ClassFossil),"Venerida","Pectinida","Ostreida","Mytilida","Myida","Crustacean")
newdat$ClassFossil[newdat$Order=="Venerida"]<-"Venerida"
newdat$ClassFossil[newdat$Order=="Pectinida"]<-"Pectinida"
newdat$ClassFossil[newdat$Order=="Ostreida"]<-"Ostreida"
newdat$ClassFossil[newdat$Order=="Mytilida"]<-"Mytilida"
newdat$ClassFossil[newdat$Order=="Myida"]<-"Myida"
newdat$ClassFossil[newdat$ClassFossil%in%c("Malacostraca","Cirripedia")]<-"Crustacean"

summary(rma(yi , vi, data=newdat,mods=~factor(Stressor1)))
summary(rma(yi , vi, data=newdat,mods=~factor(Stressor1)+factor(Class),method="REML"))

table(factor(newdat$Class)) #number of independent experiments per class
table(factor(newdat$Class[!duplicated(newdat$SpeciesAcc)]))#number of species independently tested per class. >=5 is the cutoff for inference 
meta1<-rma(yi , vi, data=newdat,mods=~factor(Stressor1)+factor(Class)+factor(Entw.stad),method="REML")
meta1<-rma(yi , vi, data=newdat,mods=~factor(Stressor1)+factor(ResponseTypeSimple),method="REML")

#By traits, motility (removing those not well represented in modern experiments)
motmod<-coef(summary(rma(yi , vi, data=newdat[!newdat$motility=="facultatively mobile",],mods=~factor(Stressor1[!newdat$motility=="facultatively mobile"])+factor(motility),method="REML")))
table(factor(newdat$motility[!duplicated(newdat$SpeciesAcc)]))
length(newdat[!newdat$motility=="facultatively mobile" & !is.na(newdat$motility),1])

#By traits, composition (removing those not well represented in modern experiments)
newdat$composition[newdat$composition%in%c("chitin","organic","phosphatic")]<-NA
compmod<-coef(summary(rma(yi , vi, data=newdat,mods=~factor(Stressor1)+factor(composition),method="REML"))) 
table(factor(newdat$composition[!duplicated(newdat$SpeciesAcc)]))
length(newdat[!is.na(newdat$composition),1])

#By traits, life_habit (removing those not well represented in modern experiments)
newdat$life_habit[newdat$life_habit=="boring"]<-NA
lifhabmod<-coef(summary(rma(yi , vi, data=newdat[!newdat$life_habit=="infaunal",],mods=~factor(Stressor1[!newdat$life_habit=="infaunal"])+factor(life_habit),method="REML"))) 
table(factor(newdat$life_habit[!duplicated(newdat$SpeciesAcc)]))
length(newdat[!newdat$life_habit=="infaunal" & !is.na(newdat$life_habit),1])

#By traits, diet (removing those not well represented in modern experiments)
newdat$diet[newdat$diet%in%c("photosymbiotic, suspension feeder","\"photoautotroph\"")]<-NA
newdat$diet[newdat$diet%in%c("suspension feeder, photosymbiotic")]<-paste("suspension feeder")
dietmod<-coef(summary(rma(yi , vi, data=newdat,mods=~factor(Stressor1)+factor(diet),method="REML"))) 
table(factor(newdat$diet[!duplicated(newdat$SpeciesAcc)]))
length(newdat[!is.na(newdat$diet),1])

#By traits, buffering
buffmod<-coef(summary(rma(yi , vi, data=newdat,mods=~factor(Stressor1)+factor(buffered),method="REML")))
table(factor(newdat$buffered[!duplicated(newdat$SpeciesAcc)]))
length(newdat[!is.na(newdat$buffered),1])

#All overlap with phylogeny highly significantly with chi-sq tests e.g. for buffering:
x<-chisq.test(newdat$buffered[!duplicated(newdat$SpeciesAcc)],newdat$ClassFossil[!duplicated(newdat$SpeciesAcc)])
x$stdres #motility***, 

#Testing the effect of fossilisation potential (FP) on reponse severity ----
newdat<-BigSubset(NA,NA,NA,"Var")              
newdat<-BigSubset(NA,NA,"T","Var")
newdat<-BigSubset(NA,NA,"pH","Var")            
newdat<-BigSubset(NA,NA,"O","Var")
newdat<-BigSubset(NA,NA,"T, O","Var") 
newdat<-BigSubset(NA,NA,"T, S","Var") 
newdat<-BigSubset(NA,NA,"pH, O","Var") 
newdat<-BigSubset(NA,NA,"T, pH","Var") 
newdat<-BigSubset(NA,NA,c("T, pH, O","T, pH, S","T, S, O"),"Var") 

#Where necessary, excluding the clusters at fp = 0 or fp = 1
summary(rma(yi[newdat$Gensamcom>0] , vi[newdat$Gensamcom>0], data=newdat,mods=~rank(Gensamcom[newdat$Gensamcom>0]))) ; length(unique(newdat$Art[is.na(newdat$Gensamcom)==F & newdat$Gensamcom>0]))
summary(rma(yi[newdat$Famsamcom>0 & newdat$Famsamcom<0.99] , vi[newdat$Famsamcom>0 & newdat$Famsamcom<0.99], data=newdat,mods=~rank(Famsamcom[newdat$Famsamcom>0 & newdat$Famsamcom<0.99]))) ;length(unique(newdat$Art[is.na(newdat$Famsamcom)==F& newdat$Famsamcom>0 & newdat$Famsamcom<0.95]))
summary(rma(yi[newdat$Ordsamcom<0.99&newdat$Ordsamcom>0] , vi[newdat$Ordsamcom<0.99&newdat$Ordsamcom>0], data=newdat,mods=~rank(Ordsamcom[newdat$Ordsamcom<0.99&newdat$Ordsamcom>0]))) ;length(unique(newdat$Art[is.na(newdat$Ordsamcom)==F & newdat$Ordsamcom<0.95&newdat$Ordsamcom>0.1]))
summary(rma(yi[newdat$Clasamcom<0.99&newdat$Clasamcom>0] , vi[newdat$Clasamcom<0.99&newdat$Clasamcom>0], data=newdat,mods=~rank(Clasamcom[newdat$Clasamcom<0.99&newdat$Clasamcom>0]))) ;length(unique(newdat$Art[is.na(newdat$Clasamcom)==F & newdat$Clasamcom<0.95&newdat$Clasamcom>0.1]))

#Extended Data Figure 4 ----
#png("FPresponses.png",height = 4,width = 5,units="in",res=300) #9)

newdat<-BigSubset(NA,NA,"pH","Var")      #order and class
clamax<-max(rank(newdat$Clasamcom[newdat$Clasamcom>0]),na.rm=T) ; ordmax<-max(rank(newdat$Ordsamcom[newdat$Ordsamcom>0 & newdat$Ordsamcom<0.99]),na.rm=T)
clasamrank<-rank(newdat$Clasamcom[newdat$Clasamcom>0])/clamax ;ordsamrank<-rank(newdat$Ordsamcom[newdat$Ordsamcom>0& newdat$Ordsamcom<0.99])/ordmax
meta1<-rma(yi[newdat$Clasamcom>0] , vi[newdat$Clasamcom>0], data=newdat,mods=~clasamrank)
length(unique(newdat$SpeciesAcc[newdat$Clasamcom>0])) #n species
O2pred<-predict(meta1,newmods = (1:clamax)/clamax)
plot(clasamrank,newdat$yi[newdat$Clasamcom>0],xlab="Taxon fossilisation potential, rank",ylab="Hedge's d SMD",main="pH responses by FP",ylim=c(-5.6,2.5))
lines((1:clamax)/clamax,O2pred[[1]],lwd=2) ;abline(h=0,col="darkgrey")
lines((1:clamax)/clamax,O2pred[[3]]);lines((1:clamax)/clamax,O2pred[[4]])
meta1<-rma(yi[newdat$Ordsamcom>0 & newdat$Ordsamcom<0.99] , vi[newdat$Ordsamcom>0 & newdat$Ordsamcom<0.99], data=newdat,mods=~ordsamrank)
length(unique(newdat$SpeciesAcc[newdat$Ordsamcom>0 & newdat$Ordsamcom<0.99]))
O2pred<-predict(meta1,newmods = (1:ordmax)/ordmax)
lines((1:ordmax)/ordmax,O2pred[[1]],col="cadetblue",lwd=2)
lines((1:ordmax)/ordmax,O2pred[[3]],col="cadetblue");lines((1:ordmax)/ordmax,O2pred[[4]],col="cadetblue") 
points(ordsamrank,newdat$yi[newdat$Ordsamcom>0 & newdat$Ordsamcom<0.99],col="cadetblue")
legend("bottomleft",legend=c("Order, P = 0.001","Class, P = 0.058"),text.col=c("cadetblue","black"),bty="n")
legend("bottomright",title="n species =",legend=c("47","67"),text.col=c("cadetblue","black"),bty="n")
#Missing value:
min(newdat$yi[newdat$Gensamcom>0],na.rm=T) ;newdat$Gensamcom[newdat$yi==min(newdat$yi[newdat$Gensamcom>0],na.rm=T)]

#dev.off()

#The effect of latitude on T&O (or other CRS) 
newdat<-BigSubset(NA,NA,"T, O","Var") 
summary(rma(yi , vi, data=newdat,mods=~abs(Latitude)))


##Complex models ----
#Proximity of starting and end conditions to the thermal niche
#temperature and oxygen
newdat<-BigSubset(NA,NA,NA,"Var") 
newdat<-BigSubset(NA,NA,"T","Var") 
newdat<-BigSubset(NA,NA,"T, O","Var") 
newdat<-BigSubset(NA,NA,"O","Var") 
newdat<-BigSubset(NA,NA,"pH","Var") 
newdat<-BigSubset(NA,NA,"T, pH","Var") 
newdat<-BigSubset(NA,NA,"pH, O","Var") 

hist(newdat$yi)

nichewidth<-newdat$TPref.Max.90th-newdat$TPref.Min.10th
vars<-cbind(abs(newdat$Latitude),newdat$O1.m,abs(newdat$O.mr),newdat$T1.m,abs(newdat$T.mr),nichewidth,newdat$Tcontprox,newdat$Ttreatprox,newdat$TPref.Max.90th,newdat$TPref.Min.10th,abs(newdat$Ph.mr),newdat$Ph1.m,newdat$t..d.) 
colnames(vars)<-c("Latitude","O1.m","O.mr","T1.m","T.mr","nichewidth","Tcontprox","Ttreatprox","TPref.Max.90th","TPref.Min.10th","Ph.mr","Ph1.m","t..d.")
cor(vars,use="pairwise.complete.obs",method="spearman")
#T
res <- glmulti(yi ~ abs(Latitude)+log(abs(T.mr)+1)+Ttreatprox+rank(t..d.), data=newdat,level=1, fitfunction=rma.glmulti, crit="aicc")
res <- glmulti(yi ~ T1.m+log(abs(T.mr)+1)+Ttreatprox+rank(TPref.Max.90th,na.last ="keep")+TPref.Min.10th+rank(t..d.), data=newdat,level=1, fitfunction=rma.glmulti, crit="aicc")
#TO
res <- glmulti(yi ~ abs(Latitude)+abs(O.mr)+log(abs(T.mr)+1)+Ttreatprox+rank(TPref.Max.90th,na.last ="keep")+log(TPref.Min.10th+5)+log(t..d.+1), data=newdat,level=1, fitfunction=rma.glmulti, crit="aicc")
res <- glmulti(yi ~ abs(O.mr)+T1.m+log(abs(T.mr)+1)+Ttreatprox+rank(TPref.Max.90th,na.last ="keep")+log(TPref.Min.10th+5)+log(t..d.+1), data=newdat,level=1, fitfunction=rma.glmulti, crit="aicc")
#O
res <- glmulti(yi ~ abs(Latitude)+abs(O.mr)+rank(Tcontprox,na.last ="keep")+rank(t..d.+1), data=newdat,level=1, fitfunction=rma.glmulti, crit="aicc")
res <- glmulti(yi ~ rank(T1.m)+abs(O.mr)+rank(Tcontprox,na.last ="keep")+rank(TPref.Max.90th,na.last ="keep")+rank(TPref.Min.10th)+rank(t..d.+1), data=newdat,level=1, fitfunction=rma.glmulti, crit="aicc")
#pH
res <- glmulti(yi ~ abs(Latitude)+log(abs(Ph.mr)+1)+Ph1.m+log(t..d.+1), data=newdat,level=1, fitfunction=rma.glmulti, crit="aicc")
res <- glmulti(yi ~ log(abs(Ph.mr)+1)+Ph1.m+rank(Tcontprox,na.last ="keep")+rank(TPref.Max.90th,na.last ="keep")+log(t..d.+1), data=newdat,level=1, fitfunction=rma.glmulti, crit="aicc")
#T,pH
res <- glmulti(yi ~ abs(Latitude)+abs(Ph.mr)+Ph1.m+log(abs(T.mr)+1)+log(t..d.+1), data=newdat,level=1, fitfunction=rma.glmulti, crit="aicc")
res <- glmulti(yi ~ abs(Ph.mr)+Ph1.m+log(abs(T.mr)+1)+rank(Ttreatprox)+rank(TPref.Max.90th,na.last ="keep")+log(t..d.+1), data=newdat,level=1, fitfunction=rma.glmulti, crit="aicc")
#pH, O
res <- glmulti(yi ~ abs(Latitude)+log(abs(Ph.mr)+1)+log(abs(O.mr)+1)+rank(TPref.Max.90th,na.last ="keep")+log(t..d.+1), data=newdat,level=1, fitfunction=rma.glmulti, crit="aicc")
res <- glmulti(yi ~ log(abs(Ph.mr)+1)+log(abs(O.mr)+1)+rank(Tcontprox)+rank(TPref.Max.90th,na.last ="keep")+log(t..d.+1), data=newdat,level=1, fitfunction=rma.glmulti, crit="aicc")

#Overview:
print(res)

#Detail on best model
summary(res@objects[[1]])

#Can look at the top 10 models for a more diffuse look at variable importance
tmp <- weightable(res)
length(grep("Latitude",tmp[1:10,1]))/length(tmp[1:10,1])
length(grep("O1.m",tmp[1:10,1]))/length(tmp[1:10,1])
length(grep("T1.m",tmp[1:10,1]))/length(tmp[1:10,1])
length(grep("T.mr",tmp[1:10,1]))/length(tmp[1:10,1])
length(grep("O.mr",tmp[1:10,1]))/length(tmp[1:10,1])
length(grep("Tcontprox",tmp[1:10,1]))/length(tmp[1:10,1])
length(grep("Ttreatprox",tmp[1:10,1]))/length(tmp[1:10,1])
length(grep("TPref.Max.90th",tmp[1:10,1]))/length(tmp[1:10,1])
length(grep("TPref.Min.10th",tmp[1:10,1]))/length(tmp[1:10,1])
length(grep("Ph.mr",tmp[1:10,1]))/length(tmp[1:10,1])
length(grep("Ph1.m",tmp[1:10,1]))/length(tmp[1:10,1])
length(grep("t..d.",tmp[1:10,1]))/length(tmp[1:10,1])

##Comparing modern and fossil responses ----

   #Experimental ranking - choose all CRS or individual combinations (only run one lot at a time)
#Either for all CRS together
newdat<-BigSubset(NA,NA,NA,"Var")     
    Stressor1<-newdat$Stressor
    levels(Stressor1)<-c(levels(Stressor1),"Trio")
    Stressor1[Stressor1 %in% c("T, pH, O","T, pH, S","T, S, O")]<-"Trio"
    levels(newdat$ClassFossil)<-c(levels(newdat$ClassFossil),"Venerida","Pectinida","Ostreida","Mytilida","Myida","Crustacean")
    newdat$ClassFossil[newdat$Order=="Venerida"]<-"Venerida"
    newdat$ClassFossil[newdat$Order=="Pectinida"]<-"Pectinida"
    newdat$ClassFossil[newdat$Order=="Ostreida"]<-"Ostreida"
    newdat$ClassFossil[newdat$Order=="Mytilida"]<-"Mytilida"
    newdat$ClassFossil[newdat$Order=="Myida"]<-"Myida"
    newdat$ClassFossil[newdat$ClassFossil%in%c("Malacostraca","Cirripedia")]<-"Crustacean"
    meta1<-rma(yi , vi, data=newdat,mods=~factor(ClassFossil)+factor(Stressor1),method="REML") 

#Or split into individual combinations (choose one of the next five lines and then run the ones defining ClassFossil, below)
newdat<-BigSubset(NA,NA,"T, pH","Var") 
newdat<-BigSubset(NA,NA,"O","Var")
#newdat<-BigSubset(NA,NA,"T, O","Var") #alternatives
#newdat<-BigSubset(NA,NA,"T","Var") 
#newdat<-BigSubset(NA,NA,"pH","Var") 
    
    levels(newdat$ClassFossil)<-c(levels(newdat$ClassFossil),"Venerida","Pectinida","Ostreida","Mytilida","Myida","Crustacean")
    newdat$ClassFossil[newdat$Order=="Venerida"]<-"Venerida"
    newdat$ClassFossil[newdat$Order=="Pectinida"]<-"Pectinida"
    newdat$ClassFossil[newdat$Order=="Ostreida"]<-"Ostreida"
    newdat$ClassFossil[newdat$Order=="Mytilida"]<-"Mytilida"
    newdat$ClassFossil[newdat$Order=="Myida"]<-"Myida"
    newdat$ClassFossil[newdat$ClassFossil%in%c("Malacostraca","Cirripedia")]<-"Crustacean"
    meta1<-rma(yi , vi, data=newdat,mods=~factor(ClassFossil),method="REML") 

#Using the following lines to summarise outputs for whichever of the different stressor combinations you chose
TaxNums<-table(factor(newdat$ClassFossil[!duplicated(newdat$SpeciesAcc)]))
TaxNums<-TaxNums[TaxNums>=4] #Minimum number for >=4 for all CRS or T,pH; adjust to >=3 for O
allres<-coef(summary(meta1))
figvals<-numeric(length=5)
for (i in 1:length(TaxNums)){
  if(length(grep(names(TaxNums)[i],row.names(allres),fixed=T))>0){
    x<-c(paste(names(TaxNums)[i]),allres[grep(names(TaxNums)[i],row.names(allres),fixed=T),c(1,5,6)],TaxNums[i])
    x[c(2,3,4)]<-unlist(x[c(2,3,4)])+allres[1,1]
    }else{
      x<-c(paste(names(TaxNums)[i]),allres[grep("intrcpt",row.names(allres),fixed=T),c(1,5,6)],TaxNums[i])}
  figvals<-rbind(figvals,x)
}
figvals<-figvals[-1,]
ExptVals<-as.data.frame(cbind(unlist(figvals[,1]),unlist(figvals[,2]),unlist(figvals[,3]),unlist(figvals[,4]),unlist(figvals[,5])))
rownames(ExptVals)<-ExptVals[,1]
ExptVals<-ExptVals[,-1]
colnames(ExptVals)<-c("Estimate","ci.lb","ci.ub","SpNum")
ExptVals[order(ExptVals[,1]),]
ExptVals<-ExptVals[order(row.names(ExptVals)),]

#Save the 'ExptVals' and 'allres[1,1]' for the different stressor combinations separately 
CRSExptVals<-ExptVals ;CRSinterct<-allres[1,1]
TpHExptVals<-ExptVals ;TpHinterct<-allres[1,1]
OExptVals<-ExptVals ;Ointerct<-allres[1,1]

   #Fossil
##Extended Data Figure 6 ----
intrank<-taxmedians[order(as.numeric(taxmedians[,2])),] 
row.names(intrank)<-intrank[,1]
intrank<-intrank[,c(2:5,1)]
intrank<-intrank[!intrank[,1]=="NaN",]
row.names(intrank)[row.names(intrank)=="Demospongea"]<-"Demospongiae"

intrank<-intrank[order(rownames(intrank)),]
cbind(rownames(intrank)[rownames(intrank) %in% rownames(CRSExptVals)],rownames(CRSExptVals)[rownames(CRSExptVals) %in% rownames(intrank)])

rownames(tax.sig.all)<-gsub("Taxa","",rownames(tax.sig.all),fixed=T) 
row.names(tax.sig.all)[row.names(tax.sig.all)=="Demospongea"]<-"Demospongiae"
PTintrank<-tax.sig.all[tax.sig.all[,4]=="51"& as.numeric(tax.sig.all[,3])<0.2,] ;modloglik[51,]##PT. 
TJintrank<-tax.sig.all[tax.sig.all[,4]=="58" & as.numeric(tax.sig.all[,3])<0.2,] ;modloglik[58,]#TJ. 
PliToaintrank<-tax.sig.all[tax.sig.all[,4]=="61" & as.numeric(tax.sig.all[,3])<0.2,] ;modloglik[61,]#PliToa. 
CenTurintrank<-tax.sig.all[tax.sig.all[,4]=="76" & as.numeric(tax.sig.all[,3])<0.2,] ;modloglik[76,]#Cenomanian-Turonian. 
PETMintrank<-tax.sig.all[tax.sig.all[,4]=="83" & as.numeric(tax.sig.all[,3])<0.2,] ;modloglik[83,]#PETM. 

#These five plots form Extended Data Figure 6
eventplot(51,"End-Permian mass extinction",PTintrank) 
eventplot(58,"End-Triassic mass extinction",TJintrank)
eventplot(61,"Pliensbachian-Toarcian boundary",PliToaintrank)
eventplot(76,"Cenomanian-Turonian boundary",CenTurintrank)
eventplot(83,"Paleocene-Eocene Thermal Maximum",PETMintrank)

##Figures ----

#Fig 1 ----
intrank<-intrank[!is.na(intrank[,1]),]
layout(matrix(1:4,2,2,byrow=T))
#rank comparison (a)
#First the correlations, for CRS
CRSbothweight<-(scale(1/as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),4])),center=F) +
                  scale(as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),4])),center=F))/2
CRScorr<-round(weightedCorr(method="spearman",
                      as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1,
                      as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),1])), 
                      weights =CRSbothweight),digits = 2)
#Correlation for T, pH
TpHbothweight<-(scale(1/as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(TpHExptVals),4])),center=F) +
                  scale(as.numeric(as.vector(TpHExptVals[rownames(TpHExptVals) %in% rownames(intrank),4])),center=F))/2
TpHcorr<-round(weightedCorr(
  rank(as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(TpHExptVals),1]))*-1),
  rank(as.numeric(as.vector(TpHExptVals[rownames(TpHExptVals) %in% rownames(intrank),1]))) ,method="spearman", 
  weights =TpHbothweight),digits = 2)
#Correlation for O
Obothweight<-(scale(1/as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(OExptVals),4])),center=F) +
                scale(as.numeric(as.vector(OExptVals[rownames(OExptVals) %in% rownames(intrank),4])),center=F))/2
Ocorr<-round(weightedCorr(
  as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(OExptVals),1]))*-1,
  as.numeric(as.vector(OExptVals[rownames(OExptVals) %in% rownames(intrank),1])) ,method="spearman", 
  weights =Obothweight),digits = 2)

pchcols<-c("#F21414","#07F80C", "#0712F8", "#F8F500", "darkgreen","#9E00F8", "#FF9C00", "#00F0FF",
           "#F00FE3", "#00B2EE") 

par(mar=c(1, 3.5,4,1) + 0.1)
plot(rank(as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1),main="",ylim=c(0.7,10),
     rank(as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),1]))),ylab="", 
     xlab="",pch=19,col=pchcols,cex=CRSbothweight*2,axes=F)# ylim=c(-1,1))
rect(0,0,11,11,col="#d1f1fd",border = NA)
points(rank(as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1),
       rank(as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),1]))),pch=19,col=pchcols,cex=CRSbothweight*2)
points(rank(as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1),
       rank(as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),1]))),pch=1,cex=CRSbothweight*2)
complm<-lm(rank(as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),1])))~rank(as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1),weights=CRSbothweight) 
abline(complm$coefficients,lty="dotted",lwd=2) 
axis(3,at=1:10);axis(2,at=1:10,las=1);box()
mtext("Increasing CRS tolerance, rank",side=2,line=2.5,cex=0.8)
mtext("Increasing genus survival odds, rank",side=3,line=2.5,cex=0.8)
labs<-rownames(intrank[rownames(intrank) %in% rownames(CRSExptVals),]) 
labs[labs=="Crustacean"]<-"Crustacea"
legend("topleft",legend=c(expression(bold("(a) Modern-fossil")),expression(bold("rank agreement"))),bty="n",cex=0.9)
legend("bottomleft",legend=c(paste("Rho = ",CRScorr,sep=""),paste(" (",paste0(round(rho.CIs(CRScorr,length(intrank[rownames(intrank) %in% rownames(CRSExptVals),1])),digits = 2),collapse=""),")",sep="")),bty="n",cex=0.9)
colslist<-cbind(labs,pchcols)

#Fossil selectivity (b)
taxnams<-c(rownames(CRSExptVals),"Brachiopoda","Cephalopoda","Chondrichthyes","Conodonta","Crinoidea","Ostracoda","Trilobita") 
taxlabs<-intrank[rownames(intrank) %in% taxnams,5][order(as.numeric(as.vector(intrank[rownames(intrank) %in% taxnams,1]))*-1)]
taxlabs[2]<-paste(taxlabs[2],"\u2020")
taxlabs[9]<-paste(taxlabs[9],"\u2020")
taxlabs[taxlabs=="Crustacean"]<-"Crustacea"
taxlabs[taxlabs=="Demospongea"]<-"Demospongiae"
#Taxa pch (open means scarce fossil record)
labpch<-numeric(length=17) ;labcol<-numeric(length=17)
labpch[1:17]<-19 ;labcol[1:17]<-"#ffffff" ;labpch[taxlabs%in%colslist[,1]]<-19 
for(i in 1:length(colslist[,1])){
  labcol[taxlabs==colslist[i,1]]<-colslist[i,2] } 

par(mar=c( 0,0, 4.1,  7.1))
plot((as.numeric(as.vector(intrank[rownames(intrank) %in% taxnams,1]))*-1)[order(as.numeric(as.vector(intrank[rownames(intrank) %in% taxnams,1]))*-1)],
     1:17,axes=F,ylab="",xlab="",xlim=c(0.55,2.6),col=labcol,main="",pch=labpch)
abline(v=seq(0.5,2.5,0.5),col="grey 94")
axis(3,labels=F) ;axis(4,at=1:17, labels=taxlabs,las=1)
arrows(as.numeric(as.vector(intrank[rownames(intrank) %in% taxnams,2]))[order(as.numeric(as.vector(intrank[rownames(intrank) %in% taxnams,1]))*-1)]*-1,1:17,
       as.numeric(as.vector(intrank[rownames(intrank) %in% taxnams,3]))[order(as.numeric(as.vector(intrank[rownames(intrank) %in% taxnams,1]))*-1)]*-1,1:17,
       lwd=1,code=3, length=0,col="#1d9a46") ;box()
points((as.numeric(as.vector(intrank[rownames(intrank) %in% taxnams,1]))*-1)[order(as.numeric(as.vector(intrank[rownames(intrank) %in% taxnams,1]))*-1)],
       1:17,col=labcol,pch=labpch)
points((as.numeric(as.vector(intrank[rownames(intrank) %in% taxnams,1]))*-1)[order(as.numeric(as.vector(intrank[rownames(intrank) %in% taxnams,1]))*-1)],1:17)
legend("topleft",legend=expression(bold("(b) Fossil selectivity")),bty="n",cex=0.9)

#Modern (c)
par(mar=c(7.1, 4.1, 0, 0))

newdat<-BigSubset(NA,NA,NA,"Var")
levels(newdat$ClassFossil)<-c(levels(newdat$ClassFossil),"Venerida","Pectinida","Ostreida","Mytilida","Myida","Hexanauplia")
newdat$ClassFossil[newdat$Order=="Venerida"]<-"Venerida"
newdat$ClassFossil[newdat$Order=="Pectinida"]<-"Pectinida"
newdat$ClassFossil[newdat$Order=="Ostreida"]<-"Ostreida"
newdat$ClassFossil[newdat$Order=="Mytilida"]<-"Mytilida"
newdat$ClassFossil[newdat$Order=="Myida"]<-"Myida"
newdat$ClassFossil[newdat$ClassFossil=="Cirripedia"]<-"Hexanauplia"
Stressor1<-newdat$Stressor
levels(Stressor1)<-c(levels(Stressor1),"Trio")
Stressor1[Stressor1 %in% c("T, pH, O","T, pH, S","T, S, O")]<-"Trio"
TaxNums<-table(factor(newdat$ClassFossil[!duplicated(newdat$SpeciesAcc)]))
TaxNums<-TaxNums[TaxNums>=4]
meta1<-rma(yi , vi, data=newdat,mods=~factor(ClassFossil)+factor(Stressor1)-1,method="REML") 
meta2<-rma(yi , vi, data=newdat,mods=~factor(ClassFossil)+factor(Stressor1),method="REML") 
allres<-coef(summary(meta1))
figvals<-numeric(length=4)
for (i in 1:length(TaxNums)){
  if(length(grep(names(TaxNums)[i],row.names(allres),fixed=T))>0){
    x<-c(paste(names(TaxNums)[i]),allres[grep(names(TaxNums)[i],row.names(allres),fixed=T),c(1,5,6)])}else{
      x<-c(paste(names(TaxNums)[i]),allres[grep("intrcpt",row.names(allres),fixed=T),c(1,5,6)])}
  figvals<-rbind(figvals,x)
}
#Next to build in the colours automatically
figvals<-cbind(figvals,rep("a",length(figvals[,1])))
for(i in 1:length(colslist[,1])){
  figvals[figvals[,1]==colslist[i,1],5]<-colslist[i,2]
  if(colslist[i,1]=="Crustacea"){figvals[figvals[,1]%in%c("Malacostraca","Hexanauplia"),5]<-colslist[i,2]}} 

figvals<-figvals[-1,]
allres2<-coef(summary(meta2))
figvals2<-numeric(length=4)
for (i in 1:length(TaxNums)){
  if(length(grep(names(TaxNums)[i],row.names(allres2),fixed=T))>0){
    x<-c(paste(names(TaxNums)[i]),allres2[grep(names(TaxNums)[i],row.names(allres2),fixed=T),c(1,5,6)])}else{
      x<-c(paste(names(TaxNums)[i]),allres2[grep("intrcpt",row.names(allres2),fixed=T),c(1,5,6)])}
  figvals2<-rbind(figvals2,x)
}
figvals2<-figvals2[-1,]
figvals2<-cbind(figvals2,c(1:3,0,4:6,6,7:9)) #old: c(1:3,0,4:6,6,7:9))

plot(1:length(figvals[,1]),figvals[,2],xlab="",ylab="",pch=NA,axes=F,ylim=c(-1.5,0.75),col="lightgrey",main="",xlim=c(1-0.4,length(unlist(figvals[,1]))+0.4)) 
abline(h=seq(-1.5,0.5,0.5),col="grey 94")
text(1:length(unlist(figvals[,1])),-1.45,labels=TaxNums[order(unlist(figvals[,2]),decreasing =F)])
arrows(1:length(unlist(figvals[,1])),unlist(figvals2[,3])[order(unlist(figvals[,2]),decreasing =F)]+c(rep(unlist(figvals[1,2]),2),0,rep(unlist(figvals[1,2]),8)),1:length(unlist(figvals[,1])),unlist(figvals2[,4])[order(unlist(figvals[,2]),decreasing =F)]+c(rep(unlist(figvals[1,2]),2),0,rep(unlist(figvals[1,2]),8)),lwd=1,code=3, length=0,col="#1980e8")
points(1:length(unlist(figvals[,1])),unlist(figvals[,2])[order(unlist(figvals[,2]),decreasing =F)],pch=19,col=unlist(figvals[,5])[order(unlist(figvals[,2]),decreasing =F)])
points(1:length(unlist(figvals[,1])),unlist(figvals[,2])[order(unlist(figvals[,2]),decreasing =F)],pch=1)
axis(1,at=1:length(unlist(figvals[,1])),labels=unlist(figvals[,1])[order(unlist(figvals[,2]),decreasing =F)],las=3)
axis(2,labels=F) ; box()

text(8.5,-1.25,"# Species represented",cex=0.9)
legend("topleft",legend=expression(bold("(c) Modern sensitivity")),bty="n",cex=0.9)

#List of correct colour/clade combinations
cbind(rownames(CRSExptVals)[rownames(CRSExptVals) %in% rownames(intrank)],pchcols)

#Modern-fossil comparison (d)
par(mar=c( 7.1,0,0, 7.1))
plot(as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1,main="",axes=F,
     as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),1])),ylab="", 
     xlab="",pch=19,col=pchcols,xlim=c(0.55,2.6),ylim=c(-1.5,0.75),cex=CRSbothweight*2)
abline(h=seq(-1.5,0.5,0.5),col="grey 94")
abline(v=seq(0.5,2.5,0.5),col="grey 94")
axis(4,las=1);axis(1,labels=T);box()
mtext("Experimental performance",side=4,line=3,cex=0.8)
mtext("Mean log-odds of genus survival",side=1,line=3,cex=0.8)
mtext(substitute(paste("mean response, Hedge's ",italic('d'))),side=4,line=4,cex=0.8)
fossilsen<-as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1
complm<-lm((as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),1])))~
             fossilsen,weights=CRSbothweight) 
abline(complm$coefficients,lty="dotted",lwd=2)
arrows(as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),2]))*-1,
       as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),1])),
       as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),3]))*-1,
       as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),1])),code = 3,angle=90,length=0,col="#1d9a46")
arrows(as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1,
       as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),2])),
       as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1,
       as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),3])),code = 3,angle=90,length=0,col="#1980e8")
points(as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1,
       as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),1])),
       pch=19,cex=CRSbothweight*2,col=pchcols)
points(as.numeric(as.vector(intrank[rownames(intrank) %in% rownames(CRSExptVals),1]))*-1,
       as.numeric(as.vector(CRSExptVals[rownames(CRSExptVals) %in% rownames(intrank),1])),
       pch=1,cex=CRSbothweight*2)
legend("topleft",legend=expression(bold("(d) Modern-fossil agreement")),bty="n",cex=0.9)


##Fig 2 (traits) ----

#modern traits
modtraits<-rbind(dietmod,lifhabmod,compmod,motmod,buffmod)
modtraits<-modtraits[-(grep(paste("Stressor1"),row.names(modtraits))),] #Remove CRS covariates
#Add relevent intercepts to coefficients (modern)
modtraits[grep("factor(diet)",row.names(modtraits),fixed=T),c(1,5,6)]<-modtraits[grep("factor(diet)",row.names(modtraits),fixed=T),c(1,5,6)]+modtraits[row.names(modtraits)=="intrcpt",1]
modtraits[grep("factor(life_habit)",row.names(modtraits),fixed=T),c(1,5,6)]<-modtraits[grep("factor(life_habit)",row.names(modtraits),fixed=T),c(1,5,6)]+modtraits[row.names(modtraits)=="intrcpt1",1]
modtraits[grep("factor(composition)",row.names(modtraits),fixed=T),c(1,5,6)]<-modtraits[grep("factor(composition)",row.names(modtraits),fixed=T),c(1,5,6)]+modtraits[row.names(modtraits)=="intrcpt2",1]
modtraits[grep("factor(motility)",row.names(modtraits),fixed=T),c(1,5,6)]<-modtraits[grep("factor(motility)",row.names(modtraits),fixed=T),c(1,5,6)]+modtraits[row.names(modtraits)=="intrcpt3",1]
modtraits[grep("factor(buffered)",row.names(modtraits),fixed=T),c(1,5,6)]<-modtraits[grep("factor(buffered)",row.names(modtraits),fixed=T),c(1,5,6)]+modtraits[row.names(modtraits)=="intrcpt4",1]
#Adjust the names (modern)
row.names(modtraits)<-sub("factor(diet)","",row.names(modtraits),fixed=T)
row.names(modtraits)<-sub("factor(life_habit)","",row.names(modtraits),fixed=T)
row.names(modtraits)<-sub("factor(composition)","",row.names(modtraits),fixed=T)
row.names(modtraits)<-sub("factor(motility)","",row.names(modtraits),fixed=T)
row.names(modtraits)<-sub("factor(buffered)TRUE","Buffered = TRUE",row.names(modtraits),fixed=T)
row.names(modtraits)<-sub("intrcpt1","epifaunal",row.names(modtraits),fixed=T)
row.names(modtraits)<-sub("intrcpt2","aragonite",row.names(modtraits),fixed=T)
row.names(modtraits)<-sub("intrcpt3","actively mobile",row.names(modtraits),fixed=T)
row.names(modtraits)<-sub("intrcpt4","buffered = FALSE",row.names(modtraits),fixed=T)
row.names(modtraits)<-sub("intrcpt","carnivore",row.names(modtraits),fixed=T)
#For labels with capital letter beginnings
#row.names(modtraits)<-c("Carnivore","Deposit feeder","Grazer","Omnivore","Suspension feeder","Epifaunal","Nektonic","Aragonite","High Mg calcite","Hydroxyapatite","Low Mg calcite","Actively mobile","Stationary","Buffered = FALSE","Buffered = TRUE")
  
#Fossil traits
fosstraits<-rbind(overalldietext,overalllifhabext,overallcompext,overallmotext,overallbuffext)
row.names(fosstraits)<-fosstraits[,1]; fosstraits<-as.data.frame(fosstraits[,-1])
colnames(fosstraits)<-c("estimate","ci.lb","ci.ub","se","AICc")
row.names(fosstraits)<-sub("diet","",row.names(fosstraits),fixed=T)
row.names(fosstraits)<-sub("life_habit","",row.names(fosstraits),fixed=T)
row.names(fosstraits)<-sub("composition","",row.names(fosstraits),fixed=T)
row.names(fosstraits)<-sub("motility","",row.names(fosstraits),fixed=T)
row.names(fosstraits)<-sub("factor(buffered)TRUE","Buffered = TRUE",row.names(fosstraits),fixed=T)
row.names(fosstraits)<-sub("factor(buffered)FALSE","Buffered = FALSE",row.names(fosstraits),fixed=T)

#Check ordering of traits
cbind(row.names(modtraits),row.names(fosstraits)[c(1:3,5,4,6:8,11,9,10,12:15)])
fosstraits<-fosstraits[c(1:3,5,4,6:8,11,9,10,12:15),]

#Fossil figure data
av.x<- -as.numeric(paste(fosstraits[,1]))-2.5
up.x<- -as.numeric(paste(fosstraits[,2]))-2.5
lo.x<- -as.numeric(paste(fosstraits[,3]))-2.5
#modern figure data
av.y<-modtraits[,1]
up.y<-modtraits[,5]
lo.y<-modtraits[,6]

#For main figure, diet and lifhab 
pdf("Traitresponses.pdf",height = 4.5,width = 5)
layout(matrix(c(1,2,2,2,2,3),6,1))
par(mai=c(0.1,0.1,0.1,0.1))
plot.new()
par(mai=c(0.05,1.4,0.05,0.5)) 
plot(av.x,(1:15)+0.1,axes=F,xlab="",ylab="",pch=16,ylim=c(7.5,0.2),xlim=c(-1.6,1),col="#1d9a46") 
rect(-1.7,8.5,1.5,5.5,col="#d1f1fd",border = NA) ; rect(-1.5,13.5,1,11.5,col="#d1f1fd",border = NA)
abline(v=c(-1,-0.5,0,0.5,1),col="grey96") ;box() 
points(av.x,(1:15)+0.1,col="#1d9a46",pch=16,cex=1.2)
arrows(up.x,(1:15)+0.1,lo.x,(1:15)+0.1,length=0,angle=90,lwd=2,col="#1d9a46")
axis(2,at=1:7,labels=row.names(modtraits)[1:7],las=1) 
points(av.y,(1:15)-0.1,col="#1980e8",pch=16,cex=1.2)
arrows(up.y,(1:15)-0.1,lo.y,(1:15)-0.1,length=0,angle=90,col="#1980e8",lwd=2)
text(c(1,1),c(5.8,0.3),labels = c(expression(bold("Life habit")),expression(bold("Diet"))),pos=2,cex=1,col="black") 
text(rep(-1.65,7),7:1,labels = c(34,74,42,21,11,14,24),font=3,pos=4,cex=1,col="darkgrey") 
text(-1.65,7.5,labels = substitute(italic("# Modern species represented")),pos=4,cex=0.9,col="darkgrey") 
abline(h=5.5) ;axis(2,at=5.5,labels="",tck=-0.1)
text(-0.48,0.3,labels = "More vulnerable",cex=0.9,pos=4,col="darkred")
arrows(-0.44,0.28,-0.75,0.28,length=0.05,angle=30,lwd=5,col="darkred",lend=1)
arrows(-0.44,0.28,-0.77,0.28,length=0.07,angle=30,lwd=2,col="darkred",lend=1)
axis(1,col="#1980e8",col.axis="#1980e8")
mtext(substitute(paste("Modern effect size, Hedge's ",italic('d')," SMD")),side=1,line=2,col="#1980e8",cex=0.7)
axis(3,at=c(-1.5,-1,-0.5,0,0.5,1),labels=c(-1.5,-1,-0.5,0,0.5,1)+2.5,col="#1d9a46",col.axis="#1d9a46")
mtext("Fossil effect size, log-odds",side=3,line=2,col="#1d9a46",cex=0.7) 
dev.off()

##Extended Data Figure 1 (traits) ----
png("TraitSMresponses.png",height = 4.5,width = 5,units = "in",res=300)
layout(matrix(c(1,2,2,2,2,2),6,1))
par(mai=c(0.1,0.1,0.1,0.1)); plot.new()
par(mai=c(1.3,1.4,0.05,0.5))
plot(av.x,(1:15)+0.1,axes=F,xlab="",ylab="",pch=16,ylim=c(16,7.6),xlim=c(-1.05,0.6),col="#1d9a46") 
rect(-1.2,13.5,1,11.5,col="#d1f1fd",border = NA)
abline(v=c(-1,-0.5,0,0.5),col="grey96") ;box() 
points(av.x,(1:15)+0.1,col="#1d9a46",pch=16,cex=1)
arrows(up.x,(1:15)+0.1,lo.x,(1:15)+0.1,length=0,angle=90,lwd=2,col="#1d9a46")
axis(2,at=8:15,labels=c(paste(row.names(modtraits)[8:13]),"Poorly buffered","Well buffered"),las=1,cex.axis=0.8)
points(av.y,(1:15)-0.1,col="#1980e8",pch=16,cex=1)
arrows(up.y,(1:15)-0.1,lo.y,(1:15)-0.1,length=0,angle=90,col="#1980e8",lwd=2)
text(c(0.6,0.6,0.6),c(7.7,11.7,13.7),labels = c(expression(bold("Skeletal composition")),expression(bold("Motility")),expression(bold("Physiological buffering"))),pos=2,cex=0.8)
text(rep(-1.1,8),8:15,labels = c(31,19,19,25,66,49,31,76),font=3,pos=4,cex=0.8,col="darkgrey")
text(-1.1,15.5,labels = "# Modern species represented",font=3,pos=4,cex=0.8,col="darkgrey")
axis(1,col="#1980e8",col.axis="#1980e8",cex.axis=0.9)
mtext(substitute(paste("Modern effect size, Hedge's ",italic('d')," SMD")),side=1,line=2,col="#1980e8",cex=0.7)
axis(3,at=c(-1,-0.5,0,0.5),labels=c(-1,-0.5,0,0.5)+2.5,col="#1d9a46",col.axis="#1d9a46",cex.axis=0.9)
mtext("Fossil effect size, log-odds",line=2,side=3,col="#1d9a46",cex=0.7) 
axis(2,at=c(13.5,11.5),labels=rep("",2),tck=0.1)
legend(-0.47,15.6,legend = "More vulnerable",bty="n",cex=0.8,col="darkred")
arrows(-0.4,16,-0.7,16,length=0.05,angle=30,lwd=5,col="darkred",lend=1)
arrows(-0.4,16,-0.72,16,length=0.07,angle=30,lwd=2,col="darkred",lend=1)
dev.off()

##Fig 3, Simple version ----
#For response types
newdat<-BigSubset(NA,NA,NA,"Var")   
Stressor1<-newdat$Stressor
levels(Stressor1)<-c(levels(Stressor1),"Trio")
Stressor1[Stressor1 %in% c("T, pH, O","T, pH, S","T, S, O")]<-"Trio"
meta1<-rma(yi , vi, data=newdat,mods=~factor(Stressor1)-1,method="REML")
meta2<-rma(yi , vi, data=newdat,mods=~factor(Stressor1),method="REML")
newdat<-BigSubset(NA,NA,NA,"Eff")   
Stressor1<-newdat$Stressor
levels(Stressor1)<-c(levels(Stressor1),"Trio")
Stressor1[Stressor1 %in% c("T, pH, O","T, pH, S","T, S, O")]<-"Trio"
meta3<-rma(yi , vi, data=newdat,mods=~factor(Stressor1)-1,method="FE")
newdat<-BigSubset(NA,NA,c("T","T, O","T, pH","T, S"),"Var",1)
meta4<-rma(yi , vi, data=newdat,mods=~factor(Stressor),method="REML")
meta5<-rma(yi , vi, data=newdat,mods=~factor(Stressor)-1,method="REML")

allres<-coef(summary(meta1))
allres2<-coef(summary(meta2))
allres3<-coef(summary(meta3))
rownames(allres)<-gsub("factor(Stressor1)","",rownames(allres),fixed=T)

#pdf(file="MMMFig3.pdf",width=6,height=5)
plot(1:length(allres[,1]),allres[,1],xlab="Treatment",ylab=substitute(paste("Performance response, Hedge's ",italic('d')," SMD")),pch=NA,axes=F,ylim=c(-3,0.6),col="lightgrey",main="",xlim=c(1-0.4,length(allres[,1])+0.4)) 
rect(4.5,-4,8.5,1.5,col="#d1f1fd",border = NA) 
rect(8.5,-4,10,1.5,col="#a4e5fd",border = NA) 
text(c(1.4,5.5,9.1),c(0.6,0.6,0.6),labels=c("One stressor","Two stressors","Three"),col="darkgrey",cex=0.8)
text(9.1,0.45,labels="stressors",col="darkgrey",cex=0.8) 
arrows(5:9,c(min(allres[c(1,5),1]),min(allres[c(2,5),1]),min(allres[c(4,5),1]),min(allres[c(1,2),1]),min(allres[c(1,2,4,5),1])),5:9,c(sum(allres[c(1,5),1]),sum(allres[c(2,5),1]),sum(allres[c(4,5),1]),sum(allres[c(1,2),1]),mean(allres[c(1,2,4,5),1])*3),code=3, length=0,lwd=20,col="#fafb81",lend=1)
arrows((5:9)-0.2,c(min(allres[c(1,5),1]),min(allres[c(2,5),1]),min(allres[c(4,5),1]),min(allres[c(1,2),1]),min(allres[c(1,2,4,5),1])),(5:9)+0.2,c(min(allres[c(1,5),1]),min(allres[c(2,5),1]),min(allres[c(4,5),1]),min(allres[c(1,2),1]),min(allres[c(1,2,4,5),1])),code=3, length=0)
arrows((5:9)-0.2,c(sum(allres[c(1,5),1]),sum(allres[c(2,5),1]),sum(allres[c(4,5),1]),sum(allres[c(1,2),1]),mean(allres[c(1,2,4,5),1])*3),(5:9)+0.2,c(sum(allres[c(1,5),1]),sum(allres[c(2,5),1]),sum(allres[c(4,5),1]),sum(allres[c(1,2),1]),mean(allres[c(1,2,4,5),1])*3),code=3, length=0)
axis(2,las=1) ;abline(h=0,col="lightgrey")

points(1:length(allres2[,1]),c(allres[c(1:2,4:5)[order(allres[c(1:2,4:5),1])],1],allres[c(3,6:8)[order(allres[c(3,6:8),1])],1],allres[9,1]),pch=19,col="#a0200b",cex=0.8)
points(1:length(allres[,1]),c(allres3[c(1:2,4:5)[order(allres[c(1:2,4:5),1])],1],allres3[c(3,6:8)[order(allres[c(3,6:8),1])],1],allres3[9,1]),pch=1,col="#a0200b",cex=0.8)
arrows(1:length(allres[,1]),c(allres2[c(1:2,4:5)[order(allres[c(1:2,4:5),1])],5],allres2[c(3,6:8)[order(allres[c(3,6:8),1])],5],allres2[9,5])+c(rep(allres[1,1],3),0,rep(allres[1,1],5)),1:length(allres[,1]),c(allres2[c(1:2,4:5)[order(allres[c(1:2,4:5),1])],6],allres2[c(3,6:8)[order(allres[c(3,6:8),1])],6],allres2[9,6])+c(rep(allres[1,1],3),0,rep(allres[1,1],5)),lwd=2,code=3, length=0,col="#a0200b")
CRSlabs<-c(rownames(allres)[c(1:2,4:5)[order(allres[c(1:2,4:5),1])]],rownames(allres)[c(3,6:8)[order(allres[c(3,6:8),1])]],rownames(allres)[9])
axis(1,at=1:length(allres[,1]),labels=CRSlabs,las=1)
abline(h=coef(meta2)[1],col="#a0200b")
abline(h=coefficients(meta3)[1],col="#a0200b",lty="dotted")

#Adds the 'T above optimum experiments only' estimates
points(c(3,5,6,7)+0.2,coef(summary(meta5))[,1],pch=19,col="#fe9602",cex=0.8) 
arrows(c(3,5,6,7)+0.2,coef(summary(meta4))[,5]+c(0,rep(coef(summary(meta4))[1,1],3)),c(3,5,6,7)+0.2,coef(summary(meta4))[,6]+c(0,rep(coef(summary(meta4))[1,1],3)),lwd=2,code=3, length=0,col="#fe9602")

text(1:9,-3,labels = c("193","39","185","66","44","211","32","32","23"),cex=0.8)

legend(0.3,-2.5,legend=c("# independent experiments"),bty="n",cex=0.8) 
legend(0.6,-2.3,legend="Warming > T optimum",lwd=2,pch=19,bty="n",lty="solid",col="#fe9602",cex=0.8,pt.cex =0.8) 
legend(6.5,-2.1,legend=c("Minimum variance","Most negative"),lwd=c(2,1),pch=c(19,1),bty="n",title="Responses prioritised by",lty=c("solid","dotted"),cex=0.8,col="#a0200b",pt.cex =0.8)
box()
#dev.off()


##Fig 4, Effect size vs latitude ----
newdat<-BigSubset(NA,NA,"T, O","Var")
levels(newdat$ClassFossil)<-c(levels(newdat$ClassFossil),"Venerida","Pectinida","Ostreida","Mytilida","Myida","Hexanauplia")
newdat$ClassFossil[newdat$Order=="Venerida"]<-"Venerida"
newdat$ClassFossil[newdat$Order=="Pectinida"]<-"Pectinida"
newdat$ClassFossil[newdat$Order=="Ostreida"]<-"Ostreida"
newdat$ClassFossil[newdat$Order=="Mytilida"]<-"Mytilida"
newdat$ClassFossil[newdat$Order=="Myida"]<-"Myida"
newdat$ClassFossil[newdat$ClassFossil=="Cirripedia"]<-"Hexanauplia"
meta1<-rma(yi , vi, data=newdat[!newdat$yi==min(newdat$yi,na.rm=T) & !newdat$yi< -6,],mods=~abs(Latitude)) 
acidpred<-predict(meta1,newmods = 5:85)

#pdf("Latresponses.pdf",height = 4.5,width = 5)
#png("LatresponsesBW.png",height = 4.5,width = 5,units = "in",res=300)

#Checking this is not driven by taxa
newlatcols<-numeric(length(newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]))
newlatcols[newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]=="Actinopterygii"]<-"#757679" 
newlatcols[newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]=="Anthozoa"]<-"#07F80C"
newlatcols[newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]=="Scyphozoa"]<-"white"
newlatcols[newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]=="Gastropoda"]<-"#FF9C00"
newlatcols[newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]=="Malacostraca"]<-"#F8F500"
newlatcols[newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]=="Mytilida"]<-"#00F0FF"
newlatcols[newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]=="Ostreida"]<-"#00B2EE"
newlatcols[newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]=="Chondrichthyes"]<-"black"

newlatpch<-numeric(length(newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]))
newlatpch[newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]=="Actinopterygii"]<-22
newlatpch[newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]=="Anthozoa"]<-24
newlatpch[newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]=="Scyphozoa"]<-23
newlatpch[newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]=="Gastropoda"]<-21
newlatpch[newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]=="Malacostraca"]<-21
newlatpch[newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]=="Mytilida"]<-22
newlatpch[newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]=="Ostreida"]<-23
newlatpch[newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]=="Chondrichthyes"]<-24

par(mai=c(1.1,1.1,0.4,0.2))
plot(abs(newdat$Latitude)[!newdat$yi==min(newdat$yi,na.rm=T)],newdat$yi[!newdat$yi==min(newdat$yi,na.rm=T)],xlab="Latitude, absolute degrees",ylab=substitute(paste("Performance response, Hedge's ",italic('d')," SMD")),main="",
     pch=newlatpch, bg=newlatcols,ylim=c(-5.8,2)) 
abline(h=0,col="darkgrey"); lines(5:85,acidpred[[1]])
lines(5:85,acidpred[[3]],lty="dashed");lines(5:85,acidpred[[4]],lty="dashed")
points(abs(newdat$Latitude)[!newdat$yi==min(newdat$yi,na.rm=T)],newdat$yi[!newdat$yi==min(newdat$yi,na.rm=T)],pch=newlatpch, bg=newlatcols)
legend("bottomleft",legend=substitute(paste("Pseudo-",italic(R^2)," = 0.301, ",italic('P'), " = 0.0003")),bty="n",cex=0.7)
legend("bottomright",cex=0.7,legend=unique(newdat$ClassFossil[!newdat$yi==min(newdat$yi,na.rm=T)]),pch=c(21,23,22,22,24,24,21,23),bty="n",pt.bg=unique(newlatcols))

#dev.off()

## Extended Data Figure 10 (Funnelplots) ----
#png("funnels.png",10,5,units="in",res=300)
layout(matrix(1:3,1,3))
newdat<-BigSubset(NA,NA,NA,"Eff")
meta1<-rma(yi , vi, data=newdat,mods=~factor(Stressor))
funnel(meta1, level=c(90, 95, 99), shade=c("white", "gray", "darkgray"),yaxis="seinv",main="(A)")
newdat<-BigSubset(NA,NA,NA,"Var")
meta1<-rma(yi , vi, data=newdat,mods=~factor(Stressor))
funnel(meta1, level=c(90, 95, 99), shade=c("white", "gray", "darkgray"),yaxis="seinv",main="(B)")
hist(randcorr[,1],xlim=c(-0.04,0.25),xlab="Effect size-error correlation, Rho",main="(C)") ;abline(v=c(0.25,median(randcorr[,1]),-0.04),col="red")
#dev.off()
